

using System;
using System.Collections.Generic;
using System.Runtime.InteropServices.WindowsRuntime;
using UnityEngine;


/// <summary>
/// 动作状态
/// </summary>
public abstract class ActionFSMState : FSMState
{
    PlayerEntity m_Role;
    public PlayerEntity role => m_Role;

    public string name => this.GetType().Name;
    
    protected ActionFSMState(FSMSystem fsm, PlayerEntity role) : base(fsm)
    {
        m_Role = role;
    }


}

/// <summary>
/// 状态机控制器
/// </summary>
public class ActionFSMSystem : FSMSystem
{
    public ActionFSMSystem() : base()
    {
        
    }
    
    public static ActionFSMSystem InitPlayerActionFSM(PlayerEntity role)
    {
        ActionFSMSystem fsm = new ActionFSMSystem();

        ActionStateNormal normal = new ActionStateNormal(fsm,role);
        ActionStateLose lose = new ActionStateLose(fsm,role);
        ActionStateWin win = new ActionStateWin(fsm,role);
        
        MeetSignTransition dropDown2Win = new MeetSignTransition(PlayerState.State_Win,normal,win);
        MeetSignTransition normal2Lose = new MeetSignTransition(PlayerState.State_Lose,normal,lose);


        //idle.AddTransition(idle2ByHit);
        normal.AddTransition(dropDown2Win);
        normal.AddTransition(normal2Lose);

        fsm.SetDefaultState(normal);
        
        return fsm;
    }
}

/// <summary>
/// 过程型动作
/// </summary>
public abstract class ProgressActionFSMState : ActionFSMState
{ 
    /// <summary>
    /// 开始时间
    /// </summary>
    protected float m_StartTime;
    /// <summary>
    /// 当前的进度时间
    /// </summary>
    public float curTime { get { return this.role.battleController.time - m_StartTime; } }

    /// <summary>
    /// 进度总时间
    /// </summary>
    protected float m_TotalTime;
    /// <summary>
    /// 进度总时间 get
    /// </summary>
    public float totalTime { get { return m_TotalTime; } }


    /// <summary>
    /// 当前进度 get
    /// </summary>
    public float progress
    {
        get
        {
            if (m_TotalTime == 0)
                return 1;

            return Mathf.Clamp01(this.curTime / m_TotalTime);
        }
    }

    /// <summary>
    /// 是否已完成 get
    /// </summary>
    public bool finished => this.curTime >= m_TotalTime;
    
    protected ProgressActionFSMState(FSMSystem fsm, PlayerEntity role) : base(fsm, role)
    {
        
    }

    public override void DoBeforeEntering()
    {
        base.DoBeforeEntering();
        
        // 记录开始时间
        m_StartTime = this.role.battleController.time;
    }

    public override void Update()
    {
        base.Update();
        
    }

}

/// <summary>
/// 加了执行进度
/// </summary>
public abstract class DoubleProgressActionFSMState : ProgressActionFSMState
{
    /// <summary>
    /// 执行时间
    /// </summary>
    protected float m_ExecuteTime;

    /// <summary>
    /// 到m_ExecuteTime的进度
    /// </summary>
    public float executeProgress
    {
        get
        {
            if (m_ExecuteTime <= 0)
            {
                return 0;
            }

            float p = Mathf.Clamp01( this.curTime / m_ExecuteTime);
            return p;
        }
    }

    /// <summary>
    /// 是否已执行
    /// </summary>
    private bool m_WaitExecute = true;
    
    protected DoubleProgressActionFSMState(FSMSystem fsm, PlayerEntity role) : base(fsm, role)
    {
        
    }

    public override void DoBeforeEntering()
    {
        base.DoBeforeEntering();   
        
        m_WaitExecute = true;

    }

    public override void Update()
    {
        base.Update();
        
        // 执行事件检测 在父类的Finish检测之前
        if (m_WaitExecute && this.curTime >= m_ExecuteTime)
        {
            this.OnExecute();
            m_WaitExecute = false;
        }
        
    }  
    
    /// <summary>
    /// 在达到执行时间时调用
    /// </summary>
    protected virtual void OnExecute()
    {
    }

}

/// <summary>
/// 正常
/// </summary>
public class ActionStateNormal : ActionFSMState
{
    public override int ID => PlayerState.State_Normal;
    public ActionStateNormal(FSMSystem fsm, PlayerEntity role) : base(fsm, role)
    {
        
    }


    public override void DoBeforeEntering()
    {
        base.DoBeforeEntering();
    }


    public override void Update()
    {
        base.Update();
       

    }

    public override void DoBeforeLeaving()
    {
        
    }
}

public class ActionStateWin : ActionFSMState
{
    public override int ID => PlayerState.State_Win;
    

    public ActionStateWin(FSMSystem fsm, PlayerEntity role) : base(fsm, role)
    {
        
    }

    public override void DoBeforeEntering()
    {
        base.DoBeforeEntering();
        
        
    }
    
    public override void Update()
    {
        base.Update();
        
    }


    public override void DoBeforeLeaving()
    {
        base.DoBeforeLeaving();
        
    }
}

public class ActionStateLose : ActionFSMState
{
    public override int ID => PlayerState.State_Lose;

    public ActionStateLose(FSMSystem fsm, PlayerEntity role) : base(fsm, role)
    {
        
        
    }


    public override void DoBeforeEntering()
    {
        base.DoBeforeEntering();
        
        
        PlayerEntity thisPlayer = this.role as PlayerEntity;
         
       
    }
    
    
    
    public override void DoBeforeLeaving()
    {
        base.DoBeforeLeaving();
    }
}





