﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEditor;

public class BuildConfigEditor : ScriptableWizard
{
    [MenuItem("Dyspace/BuildConfigEditor")]
    public static void OpenWindow()
    {
        BuildConfigEditor[] wins = Resources.FindObjectsOfTypeAll<BuildConfigEditor>();
        if (wins.Length == 0)
        {
            ScriptableWizard.DisplayWizard<BuildConfigEditor>("BuildConfigEditor");
        }
        else
        {
            wins[0].Focus();
        }
    }

    List<BuildChannelItem> itemsList = new List<BuildChannelItem>();

    void Awake()
    {
        Color c = new Color(0.5f, 0.5f, 0.5f, 0.5f);
        frame = new Texture2D(2, 2);
        frame.SetPixels(new Color[] { c, c, c, c });

    }
    private void OnEnable()
    {
        LoadData();
    }
    void OnDestroy()
    {
        if (frame != null)
        {
            DestroyImmediate(frame);
            frame = null;
        }
    }

    const string dataFilePath = "Assets/Resources/Data/ConfigChannelData.prefab";
    public void LoadData()
    {
        var data = UnityEditor.AssetDatabase.LoadAssetAtPath<ConfigChannelData>(dataFilePath);
        this.itemsList = data.channelList;
    }

    void SaveData()
    {
        for (int i = 0; i < itemsList.Count; i++)
        {
            if (string.IsNullOrEmpty(itemsList[i].channel))
            {
                EditorUtility.DisplayDialog("", string.Format("{0}项的 channel： ", i), "");
            }
        }

        EditorCommonFunctions.SavePrefab<ConfigChannelData>(dataFilePath,(data)=> { data.channelList = this.itemsList; });
    }

  

    Texture2D frame;

    void OnGUI()
    {
        float winWidth = this.position.width;
        float winHeight = this.position.height;
        
        AddPackageBtn(winWidth, winHeight);


        DrawList();
    }
    

    void PackSelected()
    {

    }
    
    void DrawList()
    {
        var list = this.itemsList;
        Rect rect;
        int i = 0;
        for (; i < list.Count; ++i)
        {
            rect = this.GetRectWithIndex(i);
            DrawFrame(rect);
            DrawItem(rect, list[i]);
        }
        DrawAddBtn(this.GetRectWithIndex(i));
    }

    const int border = 2;
    const int fieldWidth = 180;
    const int fieldHeight = 18;
    const int filedLineSpace = 3;

    const int frameWidth = 300;// fieldWidth * 2 + border * 4;
    const int frameHeight = ((fieldHeight + filedLineSpace) * BuildChannelItem.rowCount + itemTitleHeight + itemButtonHeight) + border * 4;

    Rect GetRectWithIndex(int index)
    {
        float colWidth = this.position.width;

        int oneLineCount = Mathf.FloorToInt(colWidth / (float)frameWidth);
        oneLineCount = oneLineCount < 1 ? 1 : oneLineCount;
        int rowIndex = index / oneLineCount;
        int colIndex = index % oneLineCount;
        float colOffset = 0;
        return new Rect(border * 2 + (colIndex * (frameWidth + border * 2)) + colOffset, border * 2 + rowIndex * (frameHeight + border * 2), frameWidth, frameHeight);
    }

    void DrawFrame(Rect area)
    {
        Rect top = new Rect(area.x, area.y, area.width, border);
        Rect btm = new Rect(area.x, area.y + area.height - border, area.width, border);
        Rect lft = new Rect(area.x, area.y, border, area.height);
        Rect rgt = new Rect(area.x + area.width - border, area.y, border, area.height);

        GUI.DrawTexture(top, frame);
        GUI.DrawTexture(btm, frame);
        GUI.DrawTexture(lft, frame);
        GUI.DrawTexture(rgt, frame);
    }

    const int itemTitleWidth = 100;
    const int itemTitleHeight = 30;
    const int itemButtonWidth = 100;
    const int itemButtonHeight = 30;
    void DrawItem(Rect area, BuildChannelItem item)
    {

        var titleRect = new Rect(area.x + frameWidth/2 - itemTitleWidth/2 - border * 2, area.y + border * 2 , itemTitleWidth,itemTitleHeight-6);
        item.title = EditorGUI.TextField(titleRect, item.title);

        var selectedRect = new Rect(area.x + frameWidth - itemTitleHeight - border * 2, area.y + border * 2, itemTitleHeight, itemTitleHeight);
        item.selected = EditorGUI.Toggle(selectedRect, item.selected);
        
        AddStringItemLine(area, 0, "Channel_Key:", ref item.channel);
        AddStringItemLine(area, 1, "Bundle_Id:", ref item.bundleId);
        AddStringItemLine(area, 2, "Def_Symbols:", ref item.defineSymbols);
        AddBoolItemLine(area, 3, "Debug:", ref item.debug);
        AddStringItemLine(area, 4, "KeyStore:", ref item.keystoreName);
        AddStringItemLine(area, 5, "Key_Pass:", ref item.keystorePassword);
        AddStringItemLine(area, 6, "Alias_Pass:", ref item.keystoreAliasPass);
        AddStringItemLine(area, 7, "SDK_Path:",ref item.sdkDirName);
        AddStringItemLine(area, 8, "游戏名:", ref item.appName);

        
        if (GUI.Button(new Rect(area.x + frameWidth/2 - itemButtonWidth - border * 2, area.y + frameHeight - itemButtonHeight - border * 2, itemButtonWidth, itemButtonHeight), "配置"))
        {
            BuildConfigHelper.Config(item);
        }
        if (GUI.Button(new Rect(area.x + frameWidth/2 + 10 + border * 2, area.y + frameHeight - itemButtonHeight - border * 2, itemButtonWidth, itemButtonHeight), "打包"))
        {
            BuildConfigHelper.BuildForAndroid(item);
        }
    }

    void AddStringItemLine(Rect area, int line,string title,ref string field)
    {
        var titleRect = GetFieldTitleRect(area, line);
        EditorGUI.TextArea(titleRect, title);
        var rect = GetFieldRect(area, line);

        field = (string)EditorGUI.TextField(rect, field);
    }
    void AddBoolItemLine(Rect area, int line, string title, ref bool field)
    {
        var titleRect = GetFieldTitleRect(area, line);
        EditorGUI.TextArea(titleRect, title);
        var rect = GetFieldRect(area, line);

        field = (bool)EditorGUI.Toggle(rect, field);
    }

    const int fieldTitleWidth = 80;
    Rect GetFieldTitleRect(Rect area, int line)
    {
        return GetFieldRect(area, line, fieldTitleWidth, fieldHeight, filedLineSpace,0, itemTitleHeight);
    }
    Rect GetFieldRect(Rect area, int line, int width, int height, int lineSpace,int Xoffset,int YOffset)
    {
        return new Rect(Xoffset + border * 2 + area.x, border * 2 + area.y + + fieldHeight * line + lineSpace * line+ itemTitleHeight, width, height);
    }
    Rect GetFieldRect(Rect area, int line)
    {
        return GetFieldRect(area, line, fieldWidth, fieldHeight, filedLineSpace, fieldTitleWidth + 10, itemTitleHeight);
    }

    void AddPackageBtn(float winWidth, float winHeight)
    {
        const int packButtonWidth = 100;
        const int packButtonHeight = 30;
        const int packButtonToBtm = 10;

        Rect packSelectedButtonRect = new Rect(winWidth * 0.5f - packButtonWidth -10 , winHeight - packButtonHeight - packButtonToBtm, packButtonWidth, packButtonHeight);
        if (GUI.Button(packSelectedButtonRect, "打包所选"))
        {
            PackSelected();
        }
        Rect saveButtonRect = new Rect(winWidth * 0.5f + packButtonWidth  +10, winHeight - packButtonHeight - packButtonToBtm, packButtonWidth, packButtonHeight);
        if (GUI.Button(saveButtonRect, "保存所有"))
        {
            SaveData();
        }
    }
    void DrawAddBtn(Rect area)
    {
        const int space = 10;

        area.x += space;
        area.y += space;
        area.width -= space * 2;
        area.height -= space * 2;

        if (GUI.Button(area, "+"))
        {
            this.itemsList.Add(new BuildChannelItem());
        }

    }
}
