﻿using System;
using System.Runtime.CompilerServices;
using System.Text;

public class GameDebug
{
    private enum LogLevel
    {
        LOG,
        WNG,
        ERR,
    }

    public enum LogKey
    {
        /// <summary> Message </summary>
        MSG,
        /// <summary> Trace </summary>
        TRA,
        /// <summary> UserId </summary>
        UID,
        /// <summary> UserName </summary>
        UN,
        /// <summary> UserKey </summary>
        UK,
        /// <summary> MatchingToken </summary>
        MT,
        /// <summary> RoomId </summary>
        RID,
        /// <summary> RoomCode </summary>
        RC,

    }

#if  NETCOREAPP
    static GameDebug()
    {
        var logWriter = new GameLogWriter();
        Console.SetOut(logWriter);
        Console.OutputEncoding = Encoding.UTF8;
    }
#endif

    private static void DoLog(LogLevel level, object message, string callMember, string fileName, int lineNumber,
        long? userId, string userName, string userKey, string matchingToken, string roomId, string roomCode)
    {
#if !DEBUG && !NETCOREAPP
        if (level != LogLevel.ERR)
        {
            return;
        }
#endif

        var time = System.DateTime.Now.ToString("yyyy-MM-dd HH-mm-ss-fff");

        var sb = new StringBuilder();

#if NETCOREAPP
        sb.Append($"[{time}][{level}]:[{LogKey.MSG}:{message}]\t");
#else
        sb.Append($"[{LogKey.MSG}:{message}]\t");
#endif

        if (userId.HasValue)
            sb.Append($"[{LogKey.UID}:{userId}]");
        if (!string.IsNullOrWhiteSpace(userName))
            sb.Append($"[{LogKey.UN }:{userName}]");
        if (!string.IsNullOrWhiteSpace(userKey))
            sb.Append($"[{LogKey.UK }:{userKey}]");
        if (!string.IsNullOrWhiteSpace(matchingToken))
            sb.Append($"[{LogKey.MT}:{matchingToken}]");
        if (!string.IsNullOrWhiteSpace(roomId))
            sb.Append($"[{LogKey.RID}:{roomId}]");
        if (!string.IsNullOrWhiteSpace(roomCode))
            sb.Append($"[{LogKey.RC}:{roomCode}]");

#if NETCOREAPP
        // 调用路径
        //sb.Append($"\t[{LogKey.TRA}:{fileName} line {lineNumber} {callMember}]");
        Console.WriteLine(sb.ToString());
#else
        switch (level)
        {
            case LogLevel.LOG:
                UnityEngine.Debug.Log(sb.ToString());
                break;
            case LogLevel.WNG:
                UnityEngine.Debug.LogWarning(sb.ToString());
                break;
            case LogLevel.ERR:
                UnityEngine.Debug.LogError(sb.ToString());
                break;
        }
#endif
    }

    public static void Log(object message, [CallerMemberName] string callMember = "",
        [CallerFilePath] string fileName = "", [CallerLineNumber] int lineNumber = 0, long? userId = null,
        string userName = null, string userKey = null, string matchingToken = null, string roomId = null,
        string roomCode = null)
    {
        DoLog(LogLevel.LOG, message, callMember, fileName, lineNumber, userId, userName, userKey, matchingToken, roomId, roomCode);
    }

    public static void LogWarning(object message, [CallerMemberName] string callMember = "",
        [CallerFilePath] string fileName = "", [CallerLineNumber] int lineNumber = 0, long? userId = null,
        string userName = null, string userKey = null, string matchingToken = null, string roomId = null,
        string roomCode = null)
    {
        DoLog(LogLevel.WNG, message, callMember, fileName, lineNumber, userId, userName, userKey, matchingToken, roomId, roomCode);
    }

    public static void LogError(object message, [CallerMemberName] string callMember = "",
        [CallerFilePath] string fileName = "", [CallerLineNumber] int lineNumber = 0, long? userId = null,
        string userName = null, string userKey = null, string matchingToken = null, string roomId = null,
        string roomCode = null)
    {
        DoLog(LogLevel.ERR, message, callMember, fileName, lineNumber, userId, userName, userKey, matchingToken, roomId, roomCode);
    }

#if !NETCOREAPP

    /// <summary>
    /// 光彩日志
    /// </summary>
    /// <param name="logContent">日志内容</param>
    /// <param name="color">日志颜色</param>
    public static void LogShowy(string logContent, UnityEngine.Color color)
    {
        string colorFormat = "<color=#" + UnityEngine.ColorUtility.ToHtmlStringRGB(color) + ">{0}</color>";
        UnityEngine.Debug.Log($"{string.Format(colorFormat, "[LogShowy]")}:{logContent}");

        //string colorFormat = "<color=#" + UnityEngine.ColorUtility.ToHtmlStringRGB(color) + ">{0}</color>";
        //UnityEngine.Debug.Log(string.Format(colorFormat, logContent));
    }

    /// <summary>
    /// 光彩日志
    /// </summary>
    /// <param name="logContent">日志内容</param>
    public static void LogShowy(string logContent)
    {
        LogShowy(logContent, UnityEngine.Color.green);
    }

#endif
}